import { Fragment, jsx, jsxs } from "react/jsx-runtime";
import { track } from "@tldraw/state-react";
import { modulate } from "@tldraw/utils";
import { useEffect, useState } from "react";
import { useEditor } from "../hooks/useEditor.mjs";
import { Group2d } from "../primitives/geometry/Group2d.mjs";
function useTick(isEnabled = true) {
  const [_, setTick] = useState(0);
  const editor = useEditor();
  useEffect(() => {
    if (!isEnabled) return;
    const update = () => setTick((tick) => tick + 1);
    editor.on("tick", update);
    return () => {
      editor.off("tick", update);
    };
  }, [editor, isEnabled]);
}
const GeometryDebuggingView = track(function GeometryDebuggingView2({
  showStroke = true,
  showVertices = true,
  showClosestPointOnOutline = true
}) {
  const editor = useEditor();
  useTick(showClosestPointOnOutline);
  const zoomLevel = editor.getZoomLevel();
  const renderingShapes = editor.getRenderingShapes();
  const {
    inputs: { currentPagePoint }
  } = editor;
  return /* @__PURE__ */ jsx(
    "svg",
    {
      style: {
        position: "absolute",
        pointerEvents: "none",
        zIndex: 999999999,
        top: 0,
        left: 0,
        overflow: "visible"
      },
      children: renderingShapes.map((result) => {
        const shape = editor.getShape(result.id);
        if (shape.type === "group") return null;
        const geometry = editor.getShapeGeometry(shape);
        const pageTransform = editor.getShapePageTransform(shape);
        const pointInShapeSpace = editor.getPointInShapeSpace(shape, currentPagePoint);
        const nearestPointOnShape = geometry.nearestPoint(pointInShapeSpace);
        const distanceToPoint = geometry.distanceToPoint(pointInShapeSpace, true);
        const dist = Math.abs(distanceToPoint) * zoomLevel;
        const hitInside = distanceToPoint < 0;
        const { vertices } = geometry;
        return /* @__PURE__ */ jsxs(
          "g",
          {
            transform: pageTransform.toCssString(),
            strokeLinecap: "round",
            strokeLinejoin: "round",
            children: [
              showStroke && /* @__PURE__ */ jsx(
                "g",
                {
                  stroke: geometry.debugColor ?? "red",
                  opacity: "1",
                  strokeWidth: 2 / zoomLevel,
                  fill: "none",
                  children: /* @__PURE__ */ jsx(GeometryStroke, { geometry })
                }
              ),
              showVertices && vertices.map((v, i) => /* @__PURE__ */ jsx(
                "circle",
                {
                  cx: v.x,
                  cy: v.y,
                  r: 2 / zoomLevel,
                  fill: `hsl(${modulate(i, [0, vertices.length - 1], [120, 200])}, 100%, 50%)`,
                  stroke: "black",
                  strokeWidth: 1 / zoomLevel
                },
                `v${i}`
              )),
              showClosestPointOnOutline && dist < 150 && /* @__PURE__ */ jsx(
                "line",
                {
                  x1: nearestPointOnShape.x,
                  y1: nearestPointOnShape.y,
                  x2: pointInShapeSpace.x,
                  y2: pointInShapeSpace.y,
                  opacity: 1 - dist / 150,
                  stroke: hitInside ? "goldenrod" : "dodgerblue",
                  strokeWidth: 2 / zoomLevel
                }
              )
            ]
          },
          result.id + "_outline"
        );
      })
    }
  );
});
function GeometryStroke({ geometry }) {
  if (geometry instanceof Group2d) {
    return /* @__PURE__ */ jsx(Fragment, { children: [...geometry.children, ...geometry.ignoredChildren].map((child, i) => /* @__PURE__ */ jsx(GeometryStroke, { geometry: child }, i)) });
  }
  return /* @__PURE__ */ jsx("path", { d: geometry.toSimpleSvgPath() });
}
export {
  GeometryDebuggingView
};
//# sourceMappingURL=GeometryDebuggingView.mjs.map
