import { jsx } from "react/jsx-runtime";
import { useQuickReactor, useStateTracking, useValue } from "@tldraw/state-react";
import classNames from "classnames";
import { memo, useLayoutEffect, useRef } from "react";
import { useEditor } from "../../hooks/useEditor.mjs";
import { useEditorComponents } from "../../hooks/useEditorComponents.mjs";
import { OptionalErrorBoundary } from "../ErrorBoundary.mjs";
const EvenInnererIndicator = ({ shape, util }) => {
  return useStateTracking(
    "Indicator: " + shape.type,
    () => (
      // always fetch the latest shape from the store even if the props/meta have not changed, to avoid
      // calling the render method with stale data.
      (util.indicator(util.editor.store.unsafeGetWithoutCapture(shape.id)))
    )
  );
};
const InnerIndicator = ({ editor, id }) => {
  const shape = useValue("shape for indicator", () => editor.store.get(id), [editor, id]);
  const { ShapeIndicatorErrorFallback } = useEditorComponents();
  if (!shape || shape.isLocked) return null;
  return /* @__PURE__ */ jsx(
    OptionalErrorBoundary,
    {
      fallback: ShapeIndicatorErrorFallback,
      onError: (error) => editor.annotateError(error, { origin: "react.shapeIndicator", willCrashApp: false }),
      children: /* @__PURE__ */ jsx(EvenInnererIndicator, { shape, util: editor.getShapeUtil(shape) }, shape.id)
    }
  );
};
const DefaultShapeIndicator = memo(function DefaultShapeIndicator2({
  shapeId,
  className,
  color,
  hidden,
  opacity
}) {
  const editor = useEditor();
  const rIndicator = useRef(null);
  useQuickReactor(
    "indicator transform",
    () => {
      const elm = rIndicator.current;
      if (!elm) return;
      const pageTransform = editor.getShapePageTransform(shapeId);
      if (!pageTransform) return;
      elm.style.setProperty("transform", pageTransform.toCssString());
    },
    [editor, shapeId]
  );
  useLayoutEffect(() => {
    const elm = rIndicator.current;
    if (!elm) return;
    elm.style.setProperty("display", hidden ? "none" : "block");
  }, [hidden]);
  return /* @__PURE__ */ jsx("svg", { ref: rIndicator, className: classNames("tl-overlays__item", className), children: /* @__PURE__ */ jsx("g", { className: "tl-shape-indicator", stroke: color ?? "var(--color-selected)", opacity, children: /* @__PURE__ */ jsx(InnerIndicator, { editor, id: shapeId }) }) });
});
export {
  DefaultShapeIndicator
};
//# sourceMappingURL=DefaultShapeIndicator.mjs.map
