import { atom, transact } from "@tldraw/state";
import {
  createEmptyRecordsDiff,
  isRecordsDiffEmpty,
  reverseRecordsDiff,
  squashRecordDiffsMutable
} from "@tldraw/store";
import { exhaustiveSwitchError, noop } from "@tldraw/utils";
import { uniqueId } from "../../utils/uniqueId.mjs";
import { stack } from "./Stack.mjs";
var HistoryRecorderState = /* @__PURE__ */ ((HistoryRecorderState2) => {
  HistoryRecorderState2["Recording"] = "recording";
  HistoryRecorderState2["RecordingPreserveRedoStack"] = "recordingPreserveRedoStack";
  HistoryRecorderState2["Paused"] = "paused";
  return HistoryRecorderState2;
})(HistoryRecorderState || {});
class HistoryManager {
  store;
  dispose;
  state = "recording" /* Recording */;
  pendingDiff = new PendingDiff();
  stacks = atom(
    "HistoryManager.stacks",
    {
      undos: stack(),
      redos: stack()
    },
    {
      isEqual: (a, b) => a.undos === b.undos && a.redos === b.redos
    }
  );
  annotateError;
  constructor(opts) {
    this.store = opts.store;
    this.annotateError = opts.annotateError ?? noop;
    this.dispose = this.store.addHistoryInterceptor((entry, source) => {
      if (source !== "user") return;
      switch (this.state) {
        case "recording" /* Recording */:
          this.pendingDiff.apply(entry.changes);
          this.stacks.update(({ undos }) => ({ undos, redos: stack() }));
          break;
        case "recordingPreserveRedoStack" /* RecordingPreserveRedoStack */:
          this.pendingDiff.apply(entry.changes);
          break;
        case "paused" /* Paused */:
          break;
        default:
          exhaustiveSwitchError(this.state);
      }
    });
  }
  flushPendingDiff() {
    if (this.pendingDiff.isEmpty()) return;
    const diff = this.pendingDiff.clear();
    this.stacks.update(({ undos, redos }) => ({
      undos: undos.push({ type: "diff", diff }),
      redos
    }));
  }
  getNumUndos() {
    return this.stacks.get().undos.length + (this.pendingDiff.isEmpty() ? 0 : 1);
  }
  getNumRedos() {
    return this.stacks.get().redos.length;
  }
  /** @internal */
  _isInBatch = false;
  batch = (fn, opts) => {
    const previousState = this.state;
    if (previousState !== "paused" /* Paused */ && opts?.history) {
      this.state = modeToState[opts.history];
    }
    try {
      if (this._isInBatch) {
        transact(fn);
        return this;
      }
      this._isInBatch = true;
      try {
        transact(fn);
      } catch (error) {
        this.annotateError(error);
        throw error;
      } finally {
        this._isInBatch = false;
      }
      return this;
    } finally {
      this.state = previousState;
    }
  };
  // History
  _undo = ({
    pushToRedoStack,
    toMark = void 0
  }) => {
    const previousState = this.state;
    this.state = "paused" /* Paused */;
    try {
      let { undos, redos } = this.stacks.get();
      const pendingDiff = this.pendingDiff.clear();
      const isPendingDiffEmpty = isRecordsDiffEmpty(pendingDiff);
      const diffToUndo = reverseRecordsDiff(pendingDiff);
      if (pushToRedoStack && !isPendingDiffEmpty) {
        redos = redos.push({ type: "diff", diff: pendingDiff });
      }
      let didFindMark = false;
      if (isPendingDiffEmpty) {
        while (undos.head?.type === "stop") {
          const mark = undos.head;
          undos = undos.tail;
          if (pushToRedoStack) {
            redos = redos.push(mark);
          }
          if (mark.id === toMark) {
            didFindMark = true;
            break;
          }
        }
      }
      if (!didFindMark) {
        loop: while (undos.head) {
          const undo = undos.head;
          undos = undos.tail;
          if (pushToRedoStack) {
            redos = redos.push(undo);
          }
          switch (undo.type) {
            case "diff":
              squashRecordDiffsMutable(diffToUndo, [reverseRecordsDiff(undo.diff)]);
              break;
            case "stop":
              if (!toMark) break loop;
              if (undo.id === toMark) break loop;
              break;
            default:
              exhaustiveSwitchError(undo);
          }
        }
      }
      this.store.applyDiff(diffToUndo, { ignoreEphemeralKeys: true });
      this.store.ensureStoreIsUsable();
      this.stacks.set({ undos, redos });
    } finally {
      this.state = previousState;
    }
    return this;
  };
  undo = () => {
    this._undo({ pushToRedoStack: true });
    return this;
  };
  redo = () => {
    const previousState = this.state;
    this.state = "paused" /* Paused */;
    try {
      this.flushPendingDiff();
      let { undos, redos } = this.stacks.get();
      if (redos.length === 0) {
        return this;
      }
      while (redos.head?.type === "stop") {
        undos = undos.push(redos.head);
        redos = redos.tail;
      }
      const diffToRedo = createEmptyRecordsDiff();
      while (redos.head) {
        const redo = redos.head;
        undos = undos.push(redo);
        redos = redos.tail;
        if (redo.type === "diff") {
          squashRecordDiffsMutable(diffToRedo, [redo.diff]);
        } else {
          break;
        }
      }
      this.store.applyDiff(diffToRedo, { ignoreEphemeralKeys: true });
      this.store.ensureStoreIsUsable();
      this.stacks.set({ undos, redos });
    } finally {
      this.state = previousState;
    }
    return this;
  };
  bail = () => {
    this._undo({ pushToRedoStack: false });
    return this;
  };
  bailToMark = (id) => {
    this._undo({ pushToRedoStack: false, toMark: id });
    return this;
  };
  squashToMark = (id) => {
    let top = this.stacks.get().undos;
    const popped = [];
    while (top.head && !(top.head.type === "stop" && top.head.id === id)) {
      if (top.head.type === "diff") {
        popped.push(top.head.diff);
      }
      top = top.tail;
    }
    if (!top.head || top.head?.id !== id) {
      console.error("Could not find mark to squash to: ", id);
      return this;
    }
    if (popped.length === 0) {
      return this;
    }
    const diff = createEmptyRecordsDiff();
    squashRecordDiffsMutable(diff, popped.reverse());
    this.stacks.update(({ redos }) => ({
      undos: top.push({
        type: "diff",
        diff
      }),
      redos
    }));
    return this;
  };
  mark = (id = uniqueId()) => {
    transact(() => {
      this.flushPendingDiff();
      this.stacks.update(({ undos, redos }) => ({ undos: undos.push({ type: "stop", id }), redos }));
    });
    return id;
  };
  clear() {
    this.stacks.set({ undos: stack(), redos: stack() });
    this.pendingDiff.clear();
  }
  /** @internal */
  debug() {
    const { undos, redos } = this.stacks.get();
    return {
      undos: undos.toArray(),
      redos: redos.toArray(),
      pendingDiff: this.pendingDiff.debug(),
      state: this.state
    };
  }
}
const modeToState = {
  record: "recording" /* Recording */,
  "record-preserveRedoStack": "recordingPreserveRedoStack" /* RecordingPreserveRedoStack */,
  ignore: "paused" /* Paused */
};
class PendingDiff {
  diff = createEmptyRecordsDiff();
  isEmptyAtom = atom("PendingDiff.isEmpty", true);
  clear() {
    const diff = this.diff;
    this.diff = createEmptyRecordsDiff();
    this.isEmptyAtom.set(true);
    return diff;
  }
  isEmpty() {
    return this.isEmptyAtom.get();
  }
  apply(diff) {
    squashRecordDiffsMutable(this.diff, [diff]);
    this.isEmptyAtom.set(isRecordsDiffEmpty(this.diff));
  }
  debug() {
    return { diff: this.diff, isEmpty: this.isEmpty() };
  }
}
export {
  HistoryManager
};
//# sourceMappingURL=HistoryManager.mjs.map
