import { isShapeId } from "@tldraw/tlschema";
import { structuredClone } from "@tldraw/utils";
import { Mat } from "../primitives/Mat.mjs";
import { canonicalizeRotation } from "../primitives/utils.mjs";
import { Vec } from "../primitives/Vec.mjs";
function getRotationSnapshot({ editor }) {
  const selectedShapes = editor.getSelectedShapes();
  const selectionRotation = editor.getSelectionRotation();
  const selectionBounds = editor.getSelectionRotatedPageBounds();
  const {
    inputs: { originPagePoint }
  } = editor;
  if (!selectionBounds) {
    return null;
  }
  const selectionPageCenter = selectionBounds.center.clone().rotWith(selectionBounds.point, selectionRotation);
  return {
    selectionPageCenter,
    initialCursorAngle: selectionPageCenter.angle(originPagePoint),
    initialSelectionRotation: selectionRotation,
    shapeSnapshots: selectedShapes.map((shape) => ({
      shape: structuredClone(shape),
      initialPagePoint: editor.getShapePageTransform(shape.id).point()
    }))
  };
}
function applyRotationToSnapshotShapes({
  delta,
  editor,
  snapshot,
  stage
}) {
  const { selectionPageCenter, shapeSnapshots } = snapshot;
  editor.updateShapes(
    shapeSnapshots.map(({ shape, initialPagePoint }) => {
      const parentTransform = isShapeId(shape.parentId) ? editor.getShapePageTransform(shape.parentId) : Mat.Identity();
      const newPagePoint = Vec.RotWith(initialPagePoint, selectionPageCenter, delta);
      const newLocalPoint = Mat.applyToPoint(
        // use the current parent transform in case it has moved/resized since the start
        // (e.g. if rotating a shape at the edge of a group)
        Mat.Inverse(parentTransform),
        newPagePoint
      );
      const newRotation = canonicalizeRotation(shape.rotation + delta);
      return {
        id: shape.id,
        type: shape.type,
        x: newLocalPoint.x,
        y: newLocalPoint.y,
        rotation: newRotation
      };
    })
  );
  const changes = [];
  shapeSnapshots.forEach(({ shape }) => {
    const current = editor.getShape(shape.id);
    if (!current) return;
    const util = editor.getShapeUtil(shape);
    if (stage === "start" || stage === "one-off") {
      const changeStart = util.onRotateStart?.(shape);
      if (changeStart) changes.push(changeStart);
    }
    const changeUpdate = util.onRotate?.(shape, current);
    if (changeUpdate) changes.push(changeUpdate);
    if (stage === "end" || stage === "one-off") {
      const changeEnd = util.onRotateEnd?.(shape, current);
      if (changeEnd) changes.push(changeEnd);
    }
  });
  if (changes.length > 0) {
    editor.updateShapes(changes);
  }
}
export {
  applyRotationToSnapshotShapes,
  getRotationSnapshot
};
//# sourceMappingURL=rotation.mjs.map
