import { Image } from "../network.mjs";
import { isApngAnimated } from "./apng.mjs";
import { isAvifAnimated } from "./avif.mjs";
import { isGifAnimated } from "./gif.mjs";
import { PngHelpers } from "./png.mjs";
import { isWebpAnimated } from "./webp.mjs";
const DEFAULT_SUPPORTED_VECTOR_IMAGE_TYPES = Object.freeze(["image/svg+xml"]);
const DEFAULT_SUPPORTED_STATIC_IMAGE_TYPES = Object.freeze([
  "image/jpeg",
  "image/png",
  "image/webp"
]);
const DEFAULT_SUPPORTED_ANIMATED_IMAGE_TYPES = Object.freeze([
  "image/gif",
  "image/apng",
  "image/avif"
]);
const DEFAULT_SUPPORTED_IMAGE_TYPES = Object.freeze([
  ...DEFAULT_SUPPORTED_STATIC_IMAGE_TYPES,
  ...DEFAULT_SUPPORTED_VECTOR_IMAGE_TYPES,
  ...DEFAULT_SUPPORTED_ANIMATED_IMAGE_TYPES
]);
const DEFAULT_SUPPORT_VIDEO_TYPES = Object.freeze([
  "video/mp4",
  "video/webm",
  "video/quicktime"
]);
const DEFAULT_SUPPORTED_MEDIA_TYPE_LIST = [
  ...DEFAULT_SUPPORTED_IMAGE_TYPES,
  ...DEFAULT_SUPPORT_VIDEO_TYPES
].join(",");
class MediaHelpers {
  /**
   * Load a video from a url.
   * @public
   */
  static loadVideo(src) {
    return new Promise((resolve, reject) => {
      const video = document.createElement("video");
      video.onloadeddata = () => resolve(video);
      video.onerror = (e) => {
        console.error(e);
        reject(new Error("Could not load video"));
      };
      video.crossOrigin = "anonymous";
      video.src = src;
    });
  }
  /**
   * Load an image from a url.
   * @public
   */
  static loadImage(src) {
    return new Promise((resolve, reject) => {
      const img = Image();
      img.onload = () => resolve(img);
      img.onerror = (e) => {
        console.error(e);
        reject(new Error("Could not load image"));
      };
      img.crossOrigin = "anonymous";
      img.referrerPolicy = "strict-origin-when-cross-origin";
      img.src = src;
    });
  }
  /**
   * Get the size of a video blob
   *
   * @param src - A SharedBlob containing the video
   * @public
   */
  static async getVideoSize(blob) {
    return MediaHelpers.usingObjectURL(blob, async (url) => {
      const video = await MediaHelpers.loadVideo(url);
      return { w: video.videoWidth, h: video.videoHeight };
    });
  }
  /**
   * Get the size of an image blob
   *
   * @param dataURL - A Blob containing the image.
   * @public
   */
  static async getImageSize(blob) {
    const image = await MediaHelpers.usingObjectURL(blob, MediaHelpers.loadImage);
    try {
      if (blob.type === "image/png") {
        const view = new DataView(await blob.arrayBuffer());
        if (PngHelpers.isPng(view, 0)) {
          const physChunk = PngHelpers.findChunk(view, "pHYs");
          if (physChunk) {
            const physData = PngHelpers.parsePhys(view, physChunk.dataOffset);
            if (physData.unit === 0 && physData.ppux === physData.ppuy) {
              const pixelRatio = Math.max(physData.ppux / 2834.5, 1);
              return {
                w: Math.round(image.naturalWidth / pixelRatio),
                h: Math.round(image.naturalHeight / pixelRatio)
              };
            }
          }
        }
      }
    } catch (err) {
      console.error(err);
      return { w: image.naturalWidth, h: image.naturalHeight };
    }
    return { w: image.naturalWidth, h: image.naturalHeight };
  }
  static async isAnimated(file) {
    if (file.type === "image/gif") {
      return isGifAnimated(await file.arrayBuffer());
    }
    if (file.type === "image/avif") {
      return isAvifAnimated(await file.arrayBuffer());
    }
    if (file.type === "image/webp") {
      return isWebpAnimated(await file.arrayBuffer());
    }
    if (file.type === "image/apng") {
      return isApngAnimated(await file.arrayBuffer());
    }
    return false;
  }
  static isAnimatedImageType(mimeType) {
    return DEFAULT_SUPPORTED_ANIMATED_IMAGE_TYPES.includes(mimeType || "");
  }
  static isStaticImageType(mimeType) {
    return DEFAULT_SUPPORTED_STATIC_IMAGE_TYPES.includes(mimeType || "");
  }
  static isVectorImageType(mimeType) {
    return DEFAULT_SUPPORTED_VECTOR_IMAGE_TYPES.includes(mimeType || "");
  }
  static isImageType(mimeType) {
    return DEFAULT_SUPPORTED_IMAGE_TYPES.includes(mimeType);
  }
  static async usingObjectURL(blob, fn) {
    const url = URL.createObjectURL(blob);
    try {
      return await fn(url);
    } finally {
      URL.revokeObjectURL(url);
    }
  }
}
export {
  DEFAULT_SUPPORTED_ANIMATED_IMAGE_TYPES,
  DEFAULT_SUPPORTED_IMAGE_TYPES,
  DEFAULT_SUPPORTED_MEDIA_TYPE_LIST,
  DEFAULT_SUPPORTED_STATIC_IMAGE_TYPES,
  DEFAULT_SUPPORTED_VECTOR_IMAGE_TYPES,
  DEFAULT_SUPPORT_VIDEO_TYPES,
  MediaHelpers
};
//# sourceMappingURL=media.mjs.map
