import { CommentNode } from "./comment-node";
import { ElementNode } from "./element-node";
import { EntityNode } from "./entity-node";
import { TextNode } from "./text-node";
import { HtmlNode } from "./html-node";
/**
 * @class Autolinker.htmlParser.HtmlParser
 * @extends Object
 *
 * An HTML parser implementation which simply walks an HTML string and returns an array of
 * {@link Autolinker.htmlParser.HtmlNode HtmlNodes} that represent the basic HTML structure of the input string.
 *
 * Autolinker uses this to only link URLs/emails/mentions within text nodes, effectively ignoring / "walking
 * around" HTML tags.
 */
export declare class HtmlParser {
    /**
     * Parses an HTML string and returns a simple array of {@link Autolinker.htmlParser.HtmlNode HtmlNodes}
     * to represent the HTML structure of the input string.
     *
     * @param {String} html The HTML to parse.
     * @return {Autolinker.htmlParser.HtmlNode[]}
     */
    parse(html: string): HtmlNode[];
    /**
     * Parses text and HTML entity nodes from a given string. The input string
     * should not have any HTML tags (elements) within it.
     *
     * @private
     * @param {Number} offset The offset of the text node match within the
     *   original HTML string.
     * @param {String} text The string of text to parse. This is from an HTML
     *   text node.
     * @return {Autolinker.htmlParser.HtmlNode[]} An array of HtmlNodes to
     *   represent the {@link Autolinker.htmlParser.TextNode TextNodes} and
     *   {@link Autolinker.htmlParser.EntityNode EntityNodes} found.
     */
    parseTextAndEntityNodes(offset: number, text: string): HtmlNode[];
    /**
     * Factory method to create an {@link Autolinker.htmlParser.CommentNode CommentNode}.
     *
     * @private
     * @param {Number} offset The offset of the match within the original HTML
     *   string.
     * @param {String} tagText The full text of the tag (comment) that was
     *   matched, including its &lt;!-- and --&gt;.
     * @param {String} commentText The full text of the comment that was matched.
     */
    createCommentNode(offset: number, tagText: string, commentText: string): CommentNode;
    /**
     * Factory method to create an {@link Autolinker.htmlParser.ElementNode ElementNode}.
     *
     * @private
     * @param {Number} offset The offset of the match within the original HTML
     *   string.
     * @param {String} tagText The full text of the tag (element) that was
     *   matched, including its attributes.
     * @param {String} tagName The name of the tag. Ex: An &lt;img&gt; tag would
     *   be passed to this method as "img".
     * @param {Boolean} isClosingTag `true` if it's a closing tag, false
     *   otherwise.
     * @return {Autolinker.htmlParser.ElementNode}
     */
    createElementNode(offset: number, tagText: string, tagName: string, isClosingTag: boolean): ElementNode;
    /**
     * Factory method to create a {@link Autolinker.htmlParser.EntityNode EntityNode}.
     *
     * @private
     * @param {Number} offset The offset of the match within the original HTML
     *   string.
     * @param {String} text The text that was matched for the HTML entity (such
     *   as '&amp;nbsp;').
     * @return {Autolinker.htmlParser.EntityNode}
     */
    createEntityNode(offset: number, text: string): EntityNode;
    /**
     * Factory method to create a {@link Autolinker.htmlParser.TextNode TextNode}.
     *
     * @private
     * @param {Number} offset The offset of the match within the original HTML
     *   string.
     * @param {String} text The text that was matched.
     * @return {Autolinker.htmlParser.TextNode}
     */
    createTextNode(offset: number, text: string): TextNode;
}
