import { Fragment, jsx, jsxs } from "react/jsx-runtime";
import {
  BaseBoxShapeUtil,
  Rectangle2d,
  SVGContainer,
  canonicalizeRotation,
  frameShapeMigrations,
  frameShapeProps,
  getDefaultColorTheme,
  last,
  resizeBox,
  toDomPrecision,
  useValue
} from "@tldraw/editor";
import classNames from "classnames";
import { createTextJsxFromSpans } from "../shared/createTextJsxFromSpans.mjs";
import { useDefaultColorTheme } from "../shared/useDefaultColorTheme.mjs";
import { FrameHeading } from "./components/FrameHeading.mjs";
function defaultEmptyAs(str, dflt) {
  if (str.match(/^\s*$/)) {
    return dflt;
  }
  return str;
}
class FrameShapeUtil extends BaseBoxShapeUtil {
  static type = "frame";
  static props = frameShapeProps;
  static migrations = frameShapeMigrations;
  canEdit = () => true;
  getDefaultProps() {
    return { w: 160 * 2, h: 90 * 2, name: "" };
  }
  getGeometry(shape) {
    return new Rectangle2d({
      width: shape.props.w,
      height: shape.props.h,
      isFilled: false
    });
  }
  component(shape) {
    const bounds = this.editor.getShapeGeometry(shape).bounds;
    const theme = useDefaultColorTheme();
    const isCreating = useValue(
      "is creating this shape",
      () => {
        const resizingState = this.editor.getStateDescendant("select.resizing");
        if (!resizingState) return false;
        if (!resizingState.getIsActive()) return false;
        const info = resizingState?.info;
        if (!info) return false;
        return info.isCreating && this.editor.getOnlySelectedShapeId() === shape.id;
      },
      [shape.id]
    );
    return /* @__PURE__ */ jsxs(Fragment, { children: [
      /* @__PURE__ */ jsx(SVGContainer, { children: /* @__PURE__ */ jsx(
        "rect",
        {
          className: classNames("tl-frame__body", { "tl-frame__creating": isCreating }),
          width: bounds.width,
          height: bounds.height,
          fill: theme.solid,
          stroke: theme.text
        }
      ) }),
      isCreating ? null : /* @__PURE__ */ jsx(
        FrameHeading,
        {
          id: shape.id,
          name: shape.props.name,
          width: bounds.width,
          height: bounds.height
        }
      )
    ] });
  }
  toSvg(shape, ctx) {
    const theme = getDefaultColorTheme({ isDarkMode: ctx.isDarkMode });
    const pageRotation = canonicalizeRotation(
      this.editor.getShapePageTransform(shape.id).rotation()
    );
    const offsetRotation = pageRotation + Math.PI / 4;
    const scaledRotation = (offsetRotation * (2 / Math.PI) + 4) % 4;
    const labelSide = Math.floor(scaledRotation);
    let labelTranslate;
    switch (labelSide) {
      case 0:
        labelTranslate = ``;
        break;
      case 3:
        labelTranslate = `translate(${toDomPrecision(shape.props.w)}, 0) rotate(90)`;
        break;
      case 2:
        labelTranslate = `translate(${toDomPrecision(shape.props.w)}, ${toDomPrecision(
          shape.props.h
        )}) rotate(180)`;
        break;
      case 1:
        labelTranslate = `translate(0, ${toDomPrecision(shape.props.h)}) rotate(270)`;
        break;
      default:
        throw Error("labelSide out of bounds");
    }
    const opts = {
      fontSize: 12,
      fontFamily: "Inter, sans-serif",
      textAlign: "start",
      width: shape.props.w,
      height: 32,
      padding: 0,
      lineHeight: 1,
      fontStyle: "normal",
      fontWeight: "normal",
      overflow: "truncate-ellipsis",
      verticalTextAlign: "middle"
    };
    const spans = this.editor.textMeasure.measureTextSpans(
      defaultEmptyAs(shape.props.name, "Frame") + String.fromCharCode(8203),
      opts
    );
    const firstSpan = spans[0];
    const lastSpan = last(spans);
    const labelTextWidth = lastSpan.box.w + lastSpan.box.x - firstSpan.box.x;
    const text = createTextJsxFromSpans(this.editor, spans, {
      offsetY: -opts.height - 2,
      ...opts
    });
    return /* @__PURE__ */ jsxs(Fragment, { children: [
      /* @__PURE__ */ jsx(
        "rect",
        {
          width: shape.props.w,
          height: shape.props.h,
          fill: theme.solid,
          stroke: theme.black.solid,
          strokeWidth: 1,
          rx: 1,
          ry: 1
        }
      ),
      /* @__PURE__ */ jsxs("g", { transform: labelTranslate, children: [
        /* @__PURE__ */ jsx(
          "rect",
          {
            x: -8,
            y: -opts.height - 4,
            width: labelTextWidth + 16,
            height: opts.height,
            fill: theme.background,
            rx: 4,
            ry: 4
          }
        ),
        text
      ] })
    ] });
  }
  indicator(shape) {
    const bounds = this.editor.getShapeGeometry(shape).bounds;
    return /* @__PURE__ */ jsx(
      "rect",
      {
        width: toDomPrecision(bounds.width),
        height: toDomPrecision(bounds.height),
        className: `tl-frame-indicator`
      }
    );
  }
  canReceiveNewChildrenOfType = (shape, _type) => {
    return !shape.isLocked;
  };
  providesBackgroundForChildren() {
    return true;
  }
  canDropShapes = (shape, _shapes) => {
    return !shape.isLocked;
  };
  onDragShapesOver = (frame, shapes) => {
    if (!shapes.every((child) => child.parentId === frame.id)) {
      this.editor.reparentShapes(shapes, frame.id);
    }
  };
  onDragShapesOut = (_shape, shapes) => {
    const parent = this.editor.getShape(_shape.parentId);
    const isInGroup = parent && this.editor.isShapeOfType(parent, "group");
    if (isInGroup) {
      this.editor.reparentShapes(shapes, parent.id);
    } else {
      this.editor.reparentShapes(shapes, this.editor.getCurrentPageId());
    }
  };
  onResize = (shape, info) => {
    return resizeBox(shape, info);
  };
}
export {
  FrameShapeUtil,
  defaultEmptyAs
};
//# sourceMappingURL=FrameShapeUtil.mjs.map
