import { Fragment, jsx, jsxs } from "react/jsx-runtime";
import {
  Box,
  Group2d,
  Rectangle2d,
  ShapeUtil,
  Vec,
  WeakCache,
  getDefaultColorTheme,
  noteShapeMigrations,
  noteShapeProps,
  rng,
  toDomPrecision,
  useEditor,
  useValue
} from "@tldraw/editor";
import { useCallback } from "react";
import { useCurrentTranslation } from "../../ui/hooks/useTranslation/useTranslation.mjs";
import { isRightToLeftLanguage } from "../../utils/text/text.mjs";
import { HyperlinkButton } from "../shared/HyperlinkButton.mjs";
import { SvgTextLabel } from "../shared/SvgTextLabel.mjs";
import { TextLabel } from "../shared/TextLabel.mjs";
import {
  FONT_FAMILIES,
  LABEL_FONT_SIZES,
  LABEL_PADDING,
  TEXT_PROPS
} from "../shared/default-shape-constants.mjs";
import { getFontDefForExport } from "../shared/defaultStyleDefs.mjs";
import { startEditingShapeWithLabel } from "../../tools/SelectTool/selectHelpers.mjs";
import { useDefaultColorTheme } from "../shared/useDefaultColorTheme.mjs";
import {
  CLONE_HANDLE_MARGIN,
  NOTE_CENTER_OFFSET,
  NOTE_SIZE,
  getNoteShapeForAdjacentPosition
} from "./noteHelpers.mjs";
class NoteShapeUtil extends ShapeUtil {
  static type = "note";
  static props = noteShapeProps;
  static migrations = noteShapeMigrations;
  canEdit = () => true;
  hideResizeHandles = () => true;
  hideSelectionBoundsFg = () => false;
  getDefaultProps() {
    return {
      color: "black",
      size: "m",
      text: "",
      font: "draw",
      align: "middle",
      verticalAlign: "middle",
      growY: 0,
      fontSizeAdjustment: 0,
      url: "",
      scale: 1
    };
  }
  getGeometry(shape) {
    const { labelHeight, labelWidth } = getLabelSize(this.editor, shape);
    const { scale } = shape.props;
    const lh = labelHeight * scale;
    const lw = labelWidth * scale;
    const nw = NOTE_SIZE * scale;
    const nh = getNoteHeight(shape);
    return new Group2d({
      children: [
        new Rectangle2d({ width: nw, height: nh, isFilled: true }),
        new Rectangle2d({
          x: shape.props.align === "start" ? 0 : shape.props.align === "end" ? nw - lw : (nw - lw) / 2,
          y: shape.props.verticalAlign === "start" ? 0 : shape.props.verticalAlign === "end" ? nh - lh : (nh - lh) / 2,
          width: lw,
          height: lh,
          isFilled: true,
          isLabel: true
        })
      ]
    });
  }
  getHandles(shape) {
    const { scale } = shape.props;
    const isCoarsePointer = this.editor.getInstanceState().isCoarsePointer;
    if (isCoarsePointer) return [];
    const zoom = this.editor.getZoomLevel();
    if (zoom * scale < 0.25) return [];
    const nh = getNoteHeight(shape);
    const nw = NOTE_SIZE * scale;
    const offset = CLONE_HANDLE_MARGIN / zoom * scale;
    if (zoom * scale < 0.5) {
      return [
        {
          id: "bottom",
          index: "a3",
          type: "clone",
          x: nw / 2,
          y: nh + offset
        }
      ];
    }
    return [
      {
        id: "top",
        index: "a1",
        type: "clone",
        x: nw / 2,
        y: -offset
      },
      {
        id: "right",
        index: "a2",
        type: "clone",
        x: nw + offset,
        y: nh / 2
      },
      {
        id: "bottom",
        index: "a3",
        type: "clone",
        x: nw / 2,
        y: nh + offset
      },
      {
        id: "left",
        index: "a4",
        type: "clone",
        x: -offset,
        y: nh / 2
      }
    ];
  }
  component(shape) {
    const {
      id,
      type,
      props: { scale, color, font, size, align, text, verticalAlign, fontSizeAdjustment }
    } = shape;
    const handleKeyDown = useNoteKeydownHandler(id);
    const theme = useDefaultColorTheme();
    const nw = NOTE_SIZE * scale;
    const nh = getNoteHeight(shape);
    const rotation = useValue(
      "shape rotation",
      () => this.editor.getShapePageTransform(id)?.rotation() ?? 0,
      [this.editor]
    );
    const hideShadows = useValue("zoom", () => this.editor.getZoomLevel() < 0.35 / scale, [
      scale,
      this.editor
    ]);
    const isSelected = shape.id === this.editor.getOnlySelectedShapeId();
    return /* @__PURE__ */ jsxs(Fragment, { children: [
      /* @__PURE__ */ jsx(
        "div",
        {
          id,
          className: "tl-note__container",
          style: {
            width: nw,
            height: nh,
            backgroundColor: theme[color].note.fill,
            borderBottom: hideShadows ? `${3 * scale}px solid rgb(15, 23, 31, .2)` : `none`,
            boxShadow: hideShadows ? "none" : getNoteShadow(shape.id, rotation, scale)
          },
          children: /* @__PURE__ */ jsx(
            TextLabel,
            {
              id,
              type,
              font,
              fontSize: (fontSizeAdjustment || LABEL_FONT_SIZES[size]) * scale,
              lineHeight: TEXT_PROPS.lineHeight,
              align,
              verticalAlign,
              text,
              isNote: true,
              isSelected,
              labelColor: theme[color].note.text,
              wrap: true,
              padding: 16 * scale,
              onKeyDown: handleKeyDown
            }
          )
        }
      ),
      "url" in shape.props && shape.props.url && /* @__PURE__ */ jsx(HyperlinkButton, { url: shape.props.url, zoomLevel: this.editor.getZoomLevel() })
    ] });
  }
  indicator(shape) {
    const { scale } = shape.props;
    return /* @__PURE__ */ jsx(
      "rect",
      {
        rx: scale,
        width: toDomPrecision(NOTE_SIZE * scale),
        height: toDomPrecision(getNoteHeight(shape))
      }
    );
  }
  toSvg(shape, ctx) {
    if (shape.props.text) ctx.addExportDef(getFontDefForExport(shape.props.font));
    const theme = getDefaultColorTheme({ isDarkMode: ctx.isDarkMode });
    const bounds = getBoundsForSVG(shape);
    return /* @__PURE__ */ jsxs(Fragment, { children: [
      /* @__PURE__ */ jsx("rect", { x: 5, y: 5, rx: 1, width: NOTE_SIZE - 10, height: bounds.h, fill: "rgba(0,0,0,.1)" }),
      /* @__PURE__ */ jsx(
        "rect",
        {
          rx: 1,
          width: NOTE_SIZE,
          height: bounds.h,
          fill: theme[shape.props.color].note.fill
        }
      ),
      /* @__PURE__ */ jsx(
        SvgTextLabel,
        {
          fontSize: shape.props.fontSizeAdjustment || LABEL_FONT_SIZES[shape.props.size],
          font: shape.props.font,
          align: shape.props.align,
          verticalAlign: shape.props.verticalAlign,
          text: shape.props.text,
          labelColor: theme[shape.props.color].note.text,
          bounds,
          stroke: false
        }
      )
    ] });
  }
  onBeforeCreate = (next) => {
    return getNoteSizeAdjustments(this.editor, next);
  };
  onBeforeUpdate = (prev, next) => {
    if (prev.props.text === next.props.text && prev.props.font === next.props.font && prev.props.size === next.props.size) {
      return;
    }
    return getNoteSizeAdjustments(this.editor, next);
  };
  onEditEnd = (shape) => {
    const {
      id,
      type,
      props: { text }
    } = shape;
    if (text.trimEnd() !== shape.props.text) {
      this.editor.updateShapes([
        {
          id,
          type,
          props: {
            text: text.trimEnd()
          }
        }
      ]);
    }
  };
}
function getNoteSizeAdjustments(editor, shape) {
  const { labelHeight, fontSizeAdjustment } = getLabelSize(editor, shape);
  const growY = Math.max(0, labelHeight - NOTE_SIZE);
  if (growY !== shape.props.growY || fontSizeAdjustment !== shape.props.fontSizeAdjustment) {
    return {
      ...shape,
      props: {
        ...shape.props,
        growY,
        fontSizeAdjustment
      }
    };
  }
}
function getNoteLabelSize(editor, shape) {
  const { text } = shape.props;
  if (!text) {
    const minHeight = LABEL_FONT_SIZES[shape.props.size] * TEXT_PROPS.lineHeight + LABEL_PADDING * 2;
    return { labelHeight: minHeight, labelWidth: 100, fontSizeAdjustment: 0 };
  }
  const unadjustedFontSize = LABEL_FONT_SIZES[shape.props.size];
  let fontSizeAdjustment = 0;
  let iterations = 0;
  let labelHeight = NOTE_SIZE;
  let labelWidth = NOTE_SIZE;
  const FUZZ = 1;
  do {
    fontSizeAdjustment = Math.min(unadjustedFontSize, unadjustedFontSize - iterations);
    const nextTextSize = editor.textMeasure.measureText(text, {
      ...TEXT_PROPS,
      fontFamily: FONT_FAMILIES[shape.props.font],
      fontSize: fontSizeAdjustment,
      maxWidth: NOTE_SIZE - LABEL_PADDING * 2 - FUZZ,
      disableOverflowWrapBreaking: true
    });
    labelHeight = nextTextSize.h + LABEL_PADDING * 2;
    labelWidth = nextTextSize.w + LABEL_PADDING * 2;
    if (fontSizeAdjustment <= 14) {
      const nextTextSizeWithOverflowBreak = editor.textMeasure.measureText(text, {
        ...TEXT_PROPS,
        fontFamily: FONT_FAMILIES[shape.props.font],
        fontSize: fontSizeAdjustment,
        maxWidth: NOTE_SIZE - LABEL_PADDING * 2 - FUZZ
      });
      labelHeight = nextTextSizeWithOverflowBreak.h + LABEL_PADDING * 2;
      labelWidth = nextTextSizeWithOverflowBreak.w + LABEL_PADDING * 2;
      break;
    }
    if (nextTextSize.scrollWidth.toFixed(0) === nextTextSize.w.toFixed(0)) {
      break;
    }
  } while (iterations++ < 50);
  return {
    labelHeight,
    labelWidth,
    fontSizeAdjustment
  };
}
const labelSizesForNote = new WeakCache();
function getLabelSize(editor, shape) {
  return labelSizesForNote.get(shape, () => getNoteLabelSize(editor, shape));
}
function useNoteKeydownHandler(id) {
  const editor = useEditor();
  const translation = useCurrentTranslation();
  return useCallback(
    (e) => {
      const shape = editor.getShape(id);
      if (!shape) return;
      const isTab = e.key === "Tab";
      const isCmdEnter = (e.metaKey || e.ctrlKey) && e.key === "Enter";
      if (isTab || isCmdEnter) {
        e.preventDefault();
        const pageTransform = editor.getShapePageTransform(id);
        const pageRotation = pageTransform.rotation();
        const isRTL = !!(translation.dir === "rtl" || isRightToLeftLanguage(shape.props.text));
        const offsetLength = (NOTE_SIZE + editor.options.adjacentShapeMargin + // If we're growing down, we need to account for the current shape's growY
        (isCmdEnter && !e.shiftKey ? shape.props.growY : 0)) * shape.props.scale;
        const adjacentCenter = new Vec(
          isTab ? e.shiftKey != isRTL ? -1 : 1 : 0,
          isCmdEnter ? e.shiftKey ? -1 : 1 : 0
        ).mul(offsetLength).add(NOTE_CENTER_OFFSET.clone().mul(shape.props.scale)).rot(pageRotation).add(pageTransform.point());
        const newNote = getNoteShapeForAdjacentPosition(editor, shape, adjacentCenter, pageRotation);
        if (newNote) {
          editor.mark("editing adjacent shape");
          startEditingShapeWithLabel(
            editor,
            newNote,
            true
            /* selectAll */
          );
        }
      }
    },
    [id, editor, translation.dir]
  );
}
function getNoteHeight(shape) {
  return (NOTE_SIZE + shape.props.growY) * shape.props.scale;
}
function getNoteShadow(id, rotation, scale) {
  const random = rng(id);
  const lift = Math.abs(random()) + 0.5;
  const oy = Math.cos(rotation);
  const a = 5 * scale;
  const b = 4 * scale;
  const c = 6 * scale;
  const d = 7 * scale;
  return `0px ${a - lift}px ${a}px -${a}px rgba(15, 23, 31, .6),
	0px ${(b + lift * d) * Math.max(0, oy)}px ${c + lift * d}px -${b + lift * c}px rgba(15, 23, 31, ${(0.3 + lift * 0.1).toFixed(2)}), 
	0px ${48 * scale}px ${10 * scale}px -${10 * scale}px inset rgba(15, 23, 44, ${((0.022 + random() * 5e-3) * ((1 + oy) / 2)).toFixed(2)})`;
}
function getBoundsForSVG(shape) {
  return new Box(0, 0, NOTE_SIZE, NOTE_SIZE + shape.props.growY);
}
export {
  NoteShapeUtil
};
//# sourceMappingURL=NoteShapeUtil.mjs.map
