import {
  StateNode,
  Vec,
  createShapeId
} from "@tldraw/editor";
import {
  NOTE_ADJACENT_POSITION_SNAP_RADIUS,
  getAvailableNoteAdjacentPositions
} from "../noteHelpers.mjs";
class Pointing extends StateNode {
  static id = "pointing";
  dragged = false;
  info = {};
  wasFocusedOnEnter = false;
  markId = "";
  shape = {};
  onEnter = () => {
    const { editor } = this;
    this.wasFocusedOnEnter = !editor.getIsMenuOpen();
    if (this.wasFocusedOnEnter) {
      const id = createShapeId();
      this.markId = `creating:${id}`;
      editor.mark(this.markId);
      const center = this.editor.inputs.originPagePoint.clone();
      const offset = getNoteShapeAdjacentPositionOffset(
        this.editor,
        center,
        this.editor.user.getIsDynamicResizeMode() ? 1 / this.editor.getZoomLevel() : 1
      );
      if (offset) {
        center.sub(offset);
      }
      this.shape = createNoteShape(this.editor, id, center);
    }
  };
  onPointerMove = (info) => {
    if (this.editor.inputs.isDragging) {
      if (!this.wasFocusedOnEnter) {
        const id = createShapeId();
        const center = this.editor.inputs.originPagePoint.clone();
        const offset = getNoteShapeAdjacentPositionOffset(
          this.editor,
          center,
          this.editor.user.getIsDynamicResizeMode() ? 1 / this.editor.getZoomLevel() : 1
        );
        if (offset) {
          center.sub(offset);
        }
        this.shape = createNoteShape(this.editor, id, center);
      }
      this.editor.setCurrentTool("select.translating", {
        ...info,
        target: "shape",
        shape: this.shape,
        onInteractionEnd: "note",
        isCreating: true,
        onCreate: () => {
          this.editor.setEditingShape(this.shape.id);
          this.editor.setCurrentTool("select.editing_shape");
        }
      });
    }
  };
  onPointerUp = () => {
    this.complete();
  };
  onInterrupt = () => {
    this.cancel();
  };
  onComplete = () => {
    this.complete();
  };
  onCancel = () => {
    this.cancel();
  };
  complete() {
    if (this.wasFocusedOnEnter) {
      if (this.editor.getInstanceState().isToolLocked) {
        this.parent.transition("idle");
      } else {
        this.editor.setEditingShape(this.shape.id);
        this.editor.setCurrentTool("select.editing_shape", {
          ...this.info,
          target: "shape",
          shape: this.shape
        });
      }
    }
  }
  cancel() {
    this.editor.bailToMark(this.markId);
    this.parent.transition("idle", this.info);
  }
}
function getNoteShapeAdjacentPositionOffset(editor, center, scale) {
  let min = NOTE_ADJACENT_POSITION_SNAP_RADIUS / editor.getZoomLevel();
  let offset;
  for (const pit of getAvailableNoteAdjacentPositions(editor, 0, scale, 0)) {
    const deltaToPit = Vec.Sub(center, pit);
    const dist = deltaToPit.len();
    if (dist < min) {
      min = dist;
      offset = deltaToPit;
    }
  }
  return offset;
}
function createNoteShape(editor, id, center) {
  editor.createShape({
    id,
    type: "note",
    x: center.x,
    y: center.y,
    props: {
      scale: editor.user.getIsDynamicResizeMode() ? 1 / editor.getZoomLevel() : 1
    }
  }).select(id);
  const shape = editor.getShape(id);
  const bounds = editor.getShapeGeometry(shape).bounds;
  editor.updateShapes([
    {
      id,
      type: "note",
      x: shape.x - bounds.width / 2,
      y: shape.y - bounds.height / 2
    }
  ]);
  return editor.getShape(id);
}
export {
  Pointing,
  createNoteShape,
  getNoteShapeAdjacentPositionOffset
};
//# sourceMappingURL=Pointing.mjs.map
