import {
  getPointerInfo,
  noop,
  stopEventPropagation,
  useEditor,
  useValue
} from "@tldraw/editor";
import { useCallback, useEffect, useRef } from "react";
import { INDENT, TextHelpers } from "./TextHelpers.mjs";
function useEditableText(id, type, text) {
  const editor = useEditor();
  const rInput = useRef(null);
  const isEditing = useValue("isEditing", () => editor.getEditingShapeId() === id, [editor]);
  const isEditingAnything = useValue("isEditingAnything", () => !!editor.getEditingShapeId(), [
    editor
  ]);
  useEffect(() => {
    function selectAllIfEditing({ shapeId }) {
      if (shapeId === id) {
        rInput.current?.select();
      }
    }
    editor.on("select-all-text", selectAllIfEditing);
    return () => {
      editor.off("select-all-text", selectAllIfEditing);
    };
  }, [editor, id, isEditing]);
  useEffect(() => {
    if (!isEditing) return;
    if (document.activeElement !== rInput.current) {
      rInput.current?.focus();
    }
    if (editor.getInstanceState().isCoarsePointer) {
      rInput.current?.select();
    }
    if (editor.environment.isSafari) {
      rInput.current?.blur();
      rInput.current?.focus();
    }
  }, [editor, isEditing]);
  const handleKeyDown = useCallback(
    (e) => {
      if (editor.getEditingShapeId() !== id) return;
      switch (e.key) {
        case "Enter": {
          if (e.ctrlKey || e.metaKey) {
            editor.complete();
          }
          break;
        }
      }
    },
    [editor, id]
  );
  const handleChange = useCallback(
    (e) => {
      if (editor.getEditingShapeId() !== id) return;
      let text2 = TextHelpers.normalizeText(e.currentTarget.value);
      const untabbedText = text2.replace(/\t/g, INDENT);
      if (untabbedText !== text2) {
        const selectionStart = e.currentTarget.selectionStart;
        e.currentTarget.value = untabbedText;
        e.currentTarget.selectionStart = selectionStart + (untabbedText.length - text2.length);
        e.currentTarget.selectionEnd = selectionStart + (untabbedText.length - text2.length);
        text2 = untabbedText;
      }
      editor.updateShape({
        id,
        type,
        props: { text: text2 }
      });
    },
    [editor, id, type]
  );
  const handleInputPointerDown = useCallback(
    (e) => {
      editor.dispatch({
        ...getPointerInfo(e),
        type: "pointer",
        name: "pointer_down",
        target: "shape",
        shape: editor.getShape(id)
      });
      stopEventPropagation(e);
    },
    [editor, id]
  );
  return {
    rInput,
    handleFocus: noop,
    handleBlur: noop,
    handleKeyDown,
    handleChange,
    handleInputPointerDown,
    handleDoubleClick: stopEventPropagation,
    isEmpty: text.trim().length === 0,
    isEditing,
    isEditingAnything
  };
}
export {
  useEditableText
};
//# sourceMappingURL=useEditableText.mjs.map
