import { Fragment, jsx, jsxs } from "react/jsx-runtime";
import {
  BaseBoxShapeUtil,
  HTMLContainer,
  toDomPrecision,
  useIsEditing,
  videoShapeMigrations,
  videoShapeProps
} from "@tldraw/editor";
import { useCallback, useEffect, useRef, useState } from "react";
import { BrokenAssetIcon } from "../shared/BrokenAssetIcon.mjs";
import { HyperlinkButton } from "../shared/HyperlinkButton.mjs";
import { useAsset } from "../shared/useAsset.mjs";
import { usePrefersReducedMotion } from "../shared/usePrefersReducedMotion.mjs";
class VideoShapeUtil extends BaseBoxShapeUtil {
  static type = "video";
  static props = videoShapeProps;
  static migrations = videoShapeMigrations;
  canEdit = () => true;
  isAspectRatioLocked = () => true;
  getDefaultProps() {
    return {
      w: 100,
      h: 100,
      assetId: null,
      time: 0,
      playing: true,
      url: ""
    };
  }
  component(shape) {
    const { editor } = this;
    const showControls = editor.getShapeGeometry(shape).bounds.w * editor.getZoomLevel() >= 110;
    const { asset, url } = useAsset(shape.id, shape.props.assetId, shape.props.w);
    const { time, playing } = shape.props;
    const isEditing = useIsEditing(shape.id);
    const prefersReducedMotion = usePrefersReducedMotion();
    const rVideo = useRef(null);
    const handlePlay = useCallback(
      (e) => {
        const video = e.currentTarget;
        if (!video) return;
        editor.updateShapes([
          {
            type: "video",
            id: shape.id,
            props: {
              playing: true,
              time: video.currentTime
            }
          }
        ]);
      },
      [shape.id, editor]
    );
    const handlePause = useCallback(
      (e) => {
        const video = e.currentTarget;
        if (!video) return;
        editor.updateShapes([
          {
            type: "video",
            id: shape.id,
            props: {
              playing: false,
              time: video.currentTime
            }
          }
        ]);
      },
      [shape.id, editor]
    );
    const handleSetCurrentTime = useCallback(
      (e) => {
        const video = e.currentTarget;
        if (!video) return;
        if (isEditing) {
          editor.updateShapes([
            {
              type: "video",
              id: shape.id,
              props: {
                time: video.currentTime
              }
            }
          ]);
        }
      },
      [isEditing, shape.id, editor]
    );
    const [isLoaded, setIsLoaded] = useState(false);
    const handleLoadedData = useCallback(
      (e) => {
        const video = e.currentTarget;
        if (!video) return;
        if (time !== video.currentTime) {
          video.currentTime = time;
        }
        if (!playing) {
          video.pause();
        }
        setIsLoaded(true);
      },
      [playing, time]
    );
    useEffect(() => {
      const video = rVideo.current;
      if (!video) return;
      if (isLoaded && !isEditing && time !== video.currentTime) {
        video.currentTime = time;
      }
      if (isEditing) {
        if (document.activeElement !== video) {
          video.focus();
        }
      }
    }, [isEditing, isLoaded, time]);
    useEffect(() => {
      if (prefersReducedMotion) {
        const video = rVideo.current;
        if (!video) return;
        video.pause();
        video.currentTime = 0;
      }
    }, [rVideo, prefersReducedMotion]);
    return /* @__PURE__ */ jsxs(Fragment, { children: [
      /* @__PURE__ */ jsx(
        HTMLContainer,
        {
          id: shape.id,
          style: {
            color: "var(--color-text-3)",
            backgroundColor: asset ? "transparent" : "var(--color-low)",
            border: asset ? "none" : "1px solid var(--color-low-border)"
          },
          children: /* @__PURE__ */ jsx("div", { className: "tl-counter-scaled", children: /* @__PURE__ */ jsx("div", { className: "tl-video-container", children: !asset?.props.src ? /* @__PURE__ */ jsx(BrokenAssetIcon, {}) : url ? /* @__PURE__ */ jsx(
            "video",
            {
              ref: rVideo,
              style: isEditing ? { pointerEvents: "all" } : void 0,
              className: `tl-video tl-video-shape-${shape.id.split(":")[1]}`,
              width: "100%",
              height: "100%",
              draggable: false,
              playsInline: true,
              autoPlay: true,
              muted: true,
              loop: true,
              disableRemotePlayback: true,
              disablePictureInPicture: true,
              controls: isEditing && showControls,
              onPlay: handlePlay,
              onPause: handlePause,
              onTimeUpdate: handleSetCurrentTime,
              onLoadedData: handleLoadedData,
              hidden: !isLoaded,
              children: /* @__PURE__ */ jsx("source", { src: url })
            }
          ) : null }) })
        }
      ),
      "url" in shape.props && shape.props.url && /* @__PURE__ */ jsx(HyperlinkButton, { url: shape.props.url, zoomLevel: editor.getZoomLevel() })
    ] });
  }
  indicator(shape) {
    return /* @__PURE__ */ jsx("rect", { width: toDomPrecision(shape.props.w), height: toDomPrecision(shape.props.h) });
  }
  toSvg(shape) {
    return /* @__PURE__ */ jsx("image", { href: serializeVideo(shape.id), width: shape.props.w, height: shape.props.h });
  }
}
function serializeVideo(id) {
  const splitId = id.split(":")[1];
  const video = document.querySelector(`.tl-video-shape-${splitId}`);
  if (video) {
    const canvas = document.createElement("canvas");
    canvas.width = video.videoWidth;
    canvas.height = video.videoHeight;
    canvas.getContext("2d").drawImage(video, 0, 0);
    return canvas.toDataURL("image/png");
  } else throw new Error("Video with not found when attempting serialization.");
}
export {
  VideoShapeUtil
};
//# sourceMappingURL=VideoShapeUtil.mjs.map
