import { compact } from "@tldraw/editor";
import { getOccludedChildren } from "./selectHelpers.mjs";
const INITIAL_POINTER_LAG_DURATION = 20;
const FAST_POINTER_LAG_DURATION = 100;
class DragAndDropManager {
  constructor(editor) {
    this.editor = editor;
    editor.disposables.add(this.dispose);
  }
  prevDroppingShapeId = null;
  droppingNodeTimer = null;
  first = true;
  updateDroppingNode(movingShapes, cb) {
    if (this.first) {
      this.editor.setHintingShapes(
        movingShapes.map((s) => this.editor.findShapeAncestor(s, (v) => v.type !== "group")).filter((s) => s)
      );
      this.prevDroppingShapeId = this.editor.getDroppingOverShape(this.editor.inputs.originPagePoint, movingShapes)?.id ?? null;
      this.first = false;
    }
    if (this.droppingNodeTimer === null) {
      this.setDragTimer(movingShapes, INITIAL_POINTER_LAG_DURATION, cb);
    } else if (this.editor.inputs.pointerVelocity.len() > 0.5) {
      clearTimeout(this.droppingNodeTimer);
      this.setDragTimer(movingShapes, FAST_POINTER_LAG_DURATION, cb);
    }
  }
  setDragTimer(movingShapes, duration, cb) {
    this.droppingNodeTimer = this.editor.timers.setTimeout(() => {
      this.editor.run(() => {
        this.handleDrag(this.editor.inputs.currentPagePoint, movingShapes, cb);
      });
      this.droppingNodeTimer = null;
    }, duration);
  }
  handleDrag(point, movingShapes, cb) {
    movingShapes = compact(movingShapes.map((shape) => this.editor.getShape(shape.id)));
    const nextDroppingShapeId = this.editor.getDroppingOverShape(point, movingShapes)?.id ?? null;
    if (nextDroppingShapeId === this.prevDroppingShapeId) {
      this.hintParents(movingShapes);
      return;
    }
    const { prevDroppingShapeId } = this;
    const prevDroppingShape = prevDroppingShapeId && this.editor.getShape(prevDroppingShapeId);
    const nextDroppingShape = nextDroppingShapeId && this.editor.getShape(nextDroppingShapeId);
    if (prevDroppingShape) {
      this.editor.getShapeUtil(prevDroppingShape).onDragShapesOut?.(prevDroppingShape, movingShapes);
    }
    if (nextDroppingShape) {
      this.editor.getShapeUtil(nextDroppingShape).onDragShapesOver?.(nextDroppingShape, movingShapes);
    }
    this.hintParents(movingShapes);
    cb?.();
    this.prevDroppingShapeId = nextDroppingShapeId;
  }
  hintParents(movingShapes) {
    const shapesGroupedByAncestor = /* @__PURE__ */ new Map();
    for (const shape of movingShapes) {
      const ancestor = this.editor.findShapeAncestor(shape, (v) => v.type !== "group");
      if (!ancestor) continue;
      if (!shapesGroupedByAncestor.has(ancestor.id)) {
        shapesGroupedByAncestor.set(ancestor.id, []);
      }
      shapesGroupedByAncestor.get(ancestor.id).push(shape.id);
    }
    const hintingShapes = [];
    for (const [ancestorId, shapeIds] of shapesGroupedByAncestor) {
      const ancestor = this.editor.getShape(ancestorId);
      if (!ancestor) continue;
      if (getOccludedChildren(this.editor, ancestor).length < shapeIds.length) {
        hintingShapes.push(ancestor.id);
      }
    }
    this.editor.setHintingShapes(hintingShapes);
  }
  dropShapes(shapes) {
    const { prevDroppingShapeId } = this;
    this.handleDrag(this.editor.inputs.currentPagePoint, shapes);
    if (prevDroppingShapeId) {
      const shape = this.editor.getShape(prevDroppingShapeId);
      if (!shape) return;
      this.editor.getShapeUtil(shape).onDropShapesOver?.(shape, shapes);
    }
  }
  clear() {
    this.prevDroppingShapeId = null;
    if (this.droppingNodeTimer !== null) {
      clearTimeout(this.droppingNodeTimer);
    }
    this.droppingNodeTimer = null;
    this.editor.setHintingShapes([]);
    this.first = true;
  }
  dispose = () => {
    this.clear();
  };
}
export {
  DragAndDropManager
};
//# sourceMappingURL=DragAndDropManager.mjs.map
