import { StateNode, Vec } from "@tldraw/editor";
import { getHitShapeOnCanvasPointerDown } from "../../../../selection-logic/getHitShapeOnCanvasPointerDown.mjs";
import { getTranslateCroppedImageChange } from "./crop_helpers.mjs";
class Idle extends StateNode {
  static id = "idle";
  onEnter = () => {
    this.editor.setCursor({ type: "default", rotation: 0 });
    const onlySelectedShape = this.editor.getOnlySelectedShape();
    if (onlySelectedShape) {
      this.editor.setCroppingShape(onlySelectedShape.id);
    }
  };
  onExit = () => {
    this.editor.setCursor({ type: "default", rotation: 0 });
  };
  onCancel = () => {
    this.editor.setCroppingShape(null);
    this.editor.setCurrentTool("select.idle", {});
  };
  onPointerDown = (info) => {
    if (this.editor.getIsMenuOpen()) return;
    if (info.ctrlKey) {
      this.cancel();
      this.editor.root.handleEvent(info);
      return;
    }
    switch (info.target) {
      case "canvas": {
        const hitShape = getHitShapeOnCanvasPointerDown(this.editor);
        if (hitShape && !this.editor.isShapeOfType(hitShape, "group")) {
          this.onPointerDown({
            ...info,
            shape: hitShape,
            target: "shape"
          });
          return;
        }
        this.cancel();
        this.editor.root.handleEvent(info);
        break;
      }
      case "shape": {
        if (info.shape.id === this.editor.getCroppingShapeId()) {
          this.editor.setCurrentTool("select.crop.pointing_crop", info);
          return;
        } else {
          if (this.editor.getShapeUtil(info.shape)?.canCrop(info.shape)) {
            this.editor.setCroppingShape(info.shape.id);
            this.editor.setSelectedShapes([info.shape.id]);
            this.editor.setCurrentTool("select.crop.pointing_crop", info);
          } else {
            this.cancel();
            this.editor.root.handleEvent(info);
          }
        }
        break;
      }
      case "selection": {
        switch (info.handle) {
          case "mobile_rotate":
          case "top_left_rotate":
          case "top_right_rotate":
          case "bottom_left_rotate":
          case "bottom_right_rotate": {
            this.editor.setCurrentTool("select.pointing_rotate_handle", {
              ...info,
              onInteractionEnd: "select.crop.idle"
            });
            break;
          }
          case "top":
          case "right":
          case "bottom":
          case "left":
          case "top_left":
          case "top_right":
          case "bottom_left":
          case "bottom_right": {
            this.editor.setCurrentTool("select.crop.pointing_crop_handle", {
              ...info,
              onInteractionEnd: "select.crop.idle"
            });
            break;
          }
          default: {
            this.cancel();
          }
        }
        break;
      }
    }
  };
  onDoubleClick = (info) => {
    if (this.editor.inputs.shiftKey || info.phase !== "up") return;
    const croppingShapeId = this.editor.getCroppingShapeId();
    if (!croppingShapeId) return;
    const shape = this.editor.getShape(croppingShapeId);
    if (!shape) return;
    const util = this.editor.getShapeUtil(shape);
    if (!util) return;
    if (info.target === "selection") {
      util.onDoubleClickEdge?.(shape);
      return;
    }
    this.cancel();
    this.editor.root.handleEvent(info);
  };
  onKeyDown = () => {
    this.nudgeCroppingImage(false);
  };
  onKeyRepeat = () => {
    this.nudgeCroppingImage(true);
  };
  onKeyUp = (info) => {
    switch (info.code) {
      case "Enter": {
        this.editor.setCroppingShape(null);
        this.editor.setCurrentTool("select.idle", {});
        break;
      }
    }
  };
  cancel() {
    this.editor.setCroppingShape(null);
    this.editor.setCurrentTool("select.idle", {});
  }
  nudgeCroppingImage(ephemeral = false) {
    const {
      editor: {
        inputs: { keys }
      }
    } = this;
    const shiftKey = keys.has("ShiftLeft");
    const delta = new Vec(0, 0);
    if (keys.has("ArrowLeft")) delta.x += 1;
    if (keys.has("ArrowRight")) delta.x -= 1;
    if (keys.has("ArrowUp")) delta.y += 1;
    if (keys.has("ArrowDown")) delta.y -= 1;
    if (delta.equals(new Vec(0, 0))) return;
    if (shiftKey) delta.mul(10);
    const shape = this.editor.getShape(this.editor.getCroppingShapeId());
    if (!shape) return;
    const partial = getTranslateCroppedImageChange(this.editor, shape, delta);
    if (partial) {
      if (!ephemeral) {
        this.editor.mark("translate crop");
      }
      this.editor.updateShapes([partial]);
    }
  }
}
export {
  Idle
};
//# sourceMappingURL=Idle.mjs.map
