var __defProp = Object.defineProperty;
var __getOwnPropDesc = Object.getOwnPropertyDescriptor;
var __decorateClass = (decorators, target, key, kind) => {
  var result = kind > 1 ? void 0 : kind ? __getOwnPropDesc(target, key) : target;
  for (var i = decorators.length - 1, decorator; i >= 0; i--)
    if (decorator = decorators[i])
      result = (kind ? decorator(target, key, result) : decorator(result)) || result;
  if (kind && result) __defProp(target, key, result);
  return result;
};
import {
  Box,
  Vec,
  atom,
  clamp,
  computed,
  react,
  uniqueId
} from "@tldraw/editor";
import { getRgba } from "./getRgba.mjs";
import { appendVertices, setupWebGl } from "./minimap-webgl-setup.mjs";
import { pie, rectangle, roundedRectangle } from "./minimap-webgl-shapes.mjs";
class MinimapManager {
  constructor(editor, elem, container) {
    this.editor = editor;
    this.elem = elem;
    this.container = container;
    this.gl = setupWebGl(elem);
    this.shapeGeometryCache = editor.store.createComputedCache("webgl-geometry", (r) => {
      const bounds = editor.getShapeMaskedPageBounds(r.id);
      if (!bounds) return null;
      const arr = new Float32Array(12);
      rectangle(arr, 0, bounds.x, bounds.y, bounds.w, bounds.h);
      return arr;
    });
    this.colors = this._getColors();
    this.disposables.push(this._listenForCanvasResize(), react("minimap render", this.render));
  }
  disposables = [];
  close = () => this.disposables.forEach((d) => d());
  gl;
  shapeGeometryCache;
  _getColors() {
    const style = getComputedStyle(this.editor.getContainer());
    return {
      shapeFill: getRgba(style.getPropertyValue("--color-text-3").trim()),
      selectFill: getRgba(style.getPropertyValue("--color-selected").trim()),
      viewportFill: getRgba(style.getPropertyValue("--color-muted-1").trim()),
      background: getRgba(style.getPropertyValue("--color-low").trim())
    };
  }
  colors;
  // this should be called after dark/light mode changes have propagated to the dom
  updateColors() {
    this.colors = this._getColors();
  }
  id = uniqueId();
  getDpr() {
    return this.editor.getInstanceState().devicePixelRatio;
  }
  getContentPageBounds() {
    const viewportPageBounds = this.editor.getViewportPageBounds();
    const commonShapeBounds = this.editor.getCurrentPageBounds();
    return commonShapeBounds ? Box.Expand(commonShapeBounds, viewportPageBounds) : viewportPageBounds;
  }
  getContentScreenBounds() {
    const contentPageBounds = this.getContentPageBounds();
    const topLeft = this.editor.pageToScreen(contentPageBounds.point);
    const bottomRight = this.editor.pageToScreen(
      new Vec(contentPageBounds.maxX, contentPageBounds.maxY)
    );
    return new Box(topLeft.x, topLeft.y, bottomRight.x - topLeft.x, bottomRight.y - topLeft.y);
  }
  _getCanvasBoundingRect() {
    const { x, y, width, height } = this.elem.getBoundingClientRect();
    return new Box(x, y, width, height);
  }
  canvasBoundingClientRect = atom("canvasBoundingClientRect", new Box());
  getCanvasScreenBounds() {
    return this.canvasBoundingClientRect.get();
  }
  _listenForCanvasResize() {
    const observer = new ResizeObserver(() => {
      const rect = this._getCanvasBoundingRect();
      this.canvasBoundingClientRect.set(rect);
    });
    observer.observe(this.elem);
    observer.observe(this.container);
    return () => observer.disconnect();
  }
  getCanvasSize() {
    const rect = this.canvasBoundingClientRect.get();
    const dpr = this.getDpr();
    return new Vec(rect.width * dpr, rect.height * dpr);
  }
  getCanvasClientPosition() {
    return this.canvasBoundingClientRect.get().point;
  }
  originPagePoint = new Vec();
  originPageCenter = new Vec();
  isInViewport = false;
  getCanvasPageBounds() {
    const canvasScreenBounds = this.getCanvasScreenBounds();
    const contentPageBounds = this.getContentPageBounds();
    const aspectRatio = canvasScreenBounds.width / canvasScreenBounds.height;
    let targetWidth = contentPageBounds.width;
    let targetHeight = targetWidth / aspectRatio;
    if (targetHeight < contentPageBounds.height) {
      targetHeight = contentPageBounds.height;
      targetWidth = targetHeight * aspectRatio;
    }
    const box = new Box(0, 0, targetWidth, targetHeight);
    box.center = contentPageBounds.center;
    return box;
  }
  getZoom() {
    return this.getCanvasPageBounds().width / this.getCanvasScreenBounds().width;
  }
  getCanvasPageBoundsArray() {
    const { x, y, w, h } = this.getCanvasPageBounds();
    return new Float32Array([x, y, w, h]);
  }
  getMinimapPagePoint = (clientX, clientY) => {
    const canvasPageBounds = this.getCanvasPageBounds();
    const canvasScreenBounds = this.getCanvasScreenBounds();
    let x = clientX - canvasScreenBounds.x;
    let y = clientY - canvasScreenBounds.y;
    x *= canvasPageBounds.width / canvasScreenBounds.width;
    y *= canvasPageBounds.height / canvasScreenBounds.height;
    x += canvasPageBounds.minX;
    y += canvasPageBounds.minY;
    return new Vec(x, y, 1);
  };
  minimapScreenPointToPagePoint = (x, y, shiftKey = false, clampToBounds = false) => {
    const { editor } = this;
    const vpPageBounds = editor.getViewportPageBounds();
    let { x: px, y: py } = this.getMinimapPagePoint(x, y);
    if (clampToBounds) {
      const shapesPageBounds = this.editor.getCurrentPageBounds() ?? new Box();
      const minX = shapesPageBounds.minX - vpPageBounds.width / 2;
      const maxX = shapesPageBounds.maxX + vpPageBounds.width / 2;
      const minY = shapesPageBounds.minY - vpPageBounds.height / 2;
      const maxY = shapesPageBounds.maxY + vpPageBounds.height / 2;
      const lx = Math.max(0, minX + vpPageBounds.width - px);
      const rx = Math.max(0, -(maxX - vpPageBounds.width - px));
      const ly = Math.max(0, minY + vpPageBounds.height - py);
      const ry = Math.max(0, -(maxY - vpPageBounds.height - py));
      px += (lx - rx) / 2;
      py += (ly - ry) / 2;
      px = clamp(px, minX, maxX);
      py = clamp(py, minY, maxY);
    }
    if (shiftKey) {
      const { originPagePoint } = this;
      const dx = Math.abs(px - originPagePoint.x);
      const dy = Math.abs(py - originPagePoint.y);
      if (dx > dy) {
        py = originPagePoint.y;
      } else {
        px = originPagePoint.x;
      }
    }
    return new Vec(px, py);
  };
  render = () => {
    const context = this.gl.context;
    const canvasSize = this.getCanvasSize();
    this.gl.setCanvasPageBounds(this.getCanvasPageBoundsArray());
    this.elem.width = canvasSize.x;
    this.elem.height = canvasSize.y;
    context.viewport(0, 0, canvasSize.x, canvasSize.y);
    context.clearColor(
      this.colors.background[0],
      this.colors.background[1],
      this.colors.background[2],
      1
    );
    context.clear(context.COLOR_BUFFER_BIT);
    const selectedShapes = new Set(this.editor.getSelectedShapeIds());
    const colors = this.colors;
    let selectedShapeOffset = 0;
    let unselectedShapeOffset = 0;
    const ids = this.editor.getCurrentPageShapeIdsSorted();
    for (let i = 0, len = ids.length; i < len; i++) {
      const shapeId = ids[i];
      const geometry = this.shapeGeometryCache.get(shapeId);
      if (!geometry) continue;
      const len2 = geometry.length;
      if (selectedShapes.has(shapeId)) {
        appendVertices(this.gl.selectedShapes, selectedShapeOffset, geometry);
        selectedShapeOffset += len2;
      } else {
        appendVertices(this.gl.unselectedShapes, unselectedShapeOffset, geometry);
        unselectedShapeOffset += len2;
      }
    }
    this.drawShapes(this.gl.unselectedShapes, unselectedShapeOffset, colors.shapeFill);
    this.drawShapes(this.gl.selectedShapes, selectedShapeOffset, colors.selectFill);
    this.drawViewport();
    this.drawCollaborators();
  };
  drawShapes(stuff, len, color) {
    this.gl.prepareTriangles(stuff, len);
    this.gl.setFillColor(color);
    this.gl.drawTriangles(len);
  }
  drawViewport() {
    const viewport = this.editor.getViewportPageBounds();
    const len = roundedRectangle(this.gl.viewport.vertices, viewport, 4 * this.getZoom());
    this.gl.prepareTriangles(this.gl.viewport, len);
    this.gl.setFillColor(this.colors.viewportFill);
    this.gl.drawTrianglesTransparently(len);
    if (this.editor.environment.isSafari) {
      this.gl.drawTrianglesTransparently(len);
      this.gl.drawTrianglesTransparently(len);
      this.gl.drawTrianglesTransparently(len);
    }
  }
  drawCollaborators() {
    const collaborators = this.editor.getCollaboratorsOnCurrentPage();
    if (!collaborators.length) return;
    const numSegmentsPerCircle = 20;
    const dataSizePerCircle = numSegmentsPerCircle * 6;
    const totalSize = dataSizePerCircle * collaborators.length;
    if (this.gl.collaborators.vertices.length < totalSize) {
      this.gl.collaborators.vertices = new Float32Array(totalSize);
    }
    const vertices = this.gl.collaborators.vertices;
    let offset = 0;
    const zoom = this.getZoom();
    for (const { cursor } of collaborators) {
      pie(vertices, {
        center: Vec.From(cursor),
        radius: 3 * zoom,
        offset,
        numArcSegments: numSegmentsPerCircle
      });
      offset += dataSizePerCircle;
    }
    this.gl.prepareTriangles(this.gl.collaborators, totalSize);
    offset = 0;
    for (const { color } of collaborators) {
      this.gl.setFillColor(getRgba(color));
      this.gl.context.drawArrays(this.gl.context.TRIANGLES, offset / 2, dataSizePerCircle / 2);
      offset += dataSizePerCircle;
    }
  }
}
__decorateClass([
  computed
], MinimapManager.prototype, "getDpr", 1);
__decorateClass([
  computed
], MinimapManager.prototype, "getContentPageBounds", 1);
__decorateClass([
  computed
], MinimapManager.prototype, "getContentScreenBounds", 1);
__decorateClass([
  computed
], MinimapManager.prototype, "getCanvasSize", 1);
__decorateClass([
  computed
], MinimapManager.prototype, "getCanvasClientPosition", 1);
__decorateClass([
  computed
], MinimapManager.prototype, "getCanvasPageBounds", 1);
__decorateClass([
  computed
], MinimapManager.prototype, "getZoom", 1);
__decorateClass([
  computed
], MinimapManager.prototype, "getCanvasPageBoundsArray", 1);
export {
  MinimapManager
};
//# sourceMappingURL=MinimapManager.mjs.map
