import { jsx, jsxs } from "react/jsx-runtime";
import { ContextMenuItem } from "@radix-ui/react-context-menu";
import { exhaustiveSwitchError, preventDefault } from "@tldraw/editor";
import { useState } from "react";
import { unwrapLabel } from "../../../context/actions.mjs";
import { useReadonly } from "../../../hooks/useReadonly.mjs";
import { useTranslation } from "../../../hooks/useTranslation/useTranslation.mjs";
import { kbdStr } from "../../../kbd-utils.mjs";
import { Spinner } from "../../Spinner.mjs";
import { TldrawUiButton } from "../Button/TldrawUiButton.mjs";
import { TldrawUiButtonIcon } from "../Button/TldrawUiButtonIcon.mjs";
import { TldrawUiButtonLabel } from "../Button/TldrawUiButtonLabel.mjs";
import { TldrawUiDropdownMenuItem } from "../TldrawUiDropdownMenu.mjs";
import { TldrawUiKbd } from "../TldrawUiKbd.mjs";
import { useTldrawUiMenuContext } from "./TldrawUiMenuContext.mjs";
function TldrawUiMenuItem({
  disabled = false,
  spinner = false,
  readonlyOk = false,
  id,
  kbd,
  label,
  icon,
  onSelect,
  noClose,
  isSelected
}) {
  const { type: menuType, sourceId } = useTldrawUiMenuContext();
  const msg = useTranslation();
  const [disableClicks, setDisableClicks] = useState(false);
  const isReadonlyMode = useReadonly();
  if (isReadonlyMode && !readonlyOk) return null;
  const labelToUse = unwrapLabel(label, menuType);
  const kbdTouse = kbd ? kbdStr(kbd) : void 0;
  const labelStr = labelToUse ? msg(labelToUse) : void 0;
  const titleStr = labelStr && kbdTouse ? `${labelStr} ${kbdTouse}` : labelStr;
  switch (menuType) {
    case "menu": {
      return /* @__PURE__ */ jsx(TldrawUiDropdownMenuItem, { children: /* @__PURE__ */ jsxs(
        TldrawUiButton,
        {
          type: "menu",
          "data-testid": `${sourceId}.${id}`,
          disabled,
          title: titleStr,
          onClick: (e) => {
            if (noClose) {
              preventDefault(e);
            }
            if (disableClicks) {
              setDisableClicks(false);
            } else {
              onSelect(sourceId);
            }
          },
          children: [
            /* @__PURE__ */ jsx(TldrawUiButtonLabel, { children: labelStr }),
            kbd && /* @__PURE__ */ jsx(TldrawUiKbd, { children: kbd })
          ]
        }
      ) });
    }
    case "context-menu": {
      if (disabled) return null;
      return /* @__PURE__ */ jsxs(
        ContextMenuItem,
        {
          dir: "ltr",
          title: titleStr,
          draggable: false,
          className: "tlui-button tlui-button__menu",
          "data-testid": `${sourceId}.${id}`,
          onSelect: (e) => {
            if (noClose) preventDefault(e);
            if (disableClicks) {
              setDisableClicks(false);
            } else {
              onSelect(sourceId);
            }
          },
          children: [
            /* @__PURE__ */ jsx("span", { className: "tlui-button__label", draggable: false, children: labelStr }),
            kbd && /* @__PURE__ */ jsx(TldrawUiKbd, { children: kbd }),
            spinner && /* @__PURE__ */ jsx(Spinner, {})
          ]
        }
      );
    }
    case "panel": {
      return /* @__PURE__ */ jsxs(
        TldrawUiButton,
        {
          "data-testid": `${sourceId}.${id}`,
          type: "menu",
          title: titleStr,
          disabled,
          onClick: () => onSelect(sourceId),
          children: [
            /* @__PURE__ */ jsx(TldrawUiButtonLabel, { children: labelStr }),
            icon && /* @__PURE__ */ jsx(TldrawUiButtonIcon, { icon })
          ]
        }
      );
    }
    case "small-icons":
    case "icons": {
      return /* @__PURE__ */ jsx(
        TldrawUiButton,
        {
          "data-testid": `${sourceId}.${id}`,
          type: "icon",
          title: titleStr,
          disabled,
          onClick: () => onSelect(sourceId),
          children: /* @__PURE__ */ jsx(TldrawUiButtonIcon, { icon, small: menuType === "small-icons" })
        }
      );
    }
    case "keyboard-shortcuts": {
      if (!kbd) {
        console.warn(
          `Menu item '${label}' isn't shown in the keyboard shortcuts dialog because it doesn't have a keyboard shortcut.`
        );
        return null;
      }
      return /* @__PURE__ */ jsxs("div", { className: "tlui-shortcuts-dialog__key-pair", "data-testid": `${sourceId}.${id}`, children: [
        /* @__PURE__ */ jsx("div", { className: "tlui-shortcuts-dialog__key-pair__key", children: labelStr }),
        /* @__PURE__ */ jsx("div", { className: "tlui-shortcuts-dialog__key-pair__value", children: /* @__PURE__ */ jsx(TldrawUiKbd, { visibleOnMobileLayout: true, children: kbd }) })
      ] });
    }
    case "helper-buttons": {
      return /* @__PURE__ */ jsxs(TldrawUiButton, { type: "low", onClick: () => onSelect(sourceId), children: [
        /* @__PURE__ */ jsx(TldrawUiButtonIcon, { icon }),
        /* @__PURE__ */ jsx(TldrawUiButtonLabel, { children: labelStr })
      ] });
    }
    case "toolbar": {
      return /* @__PURE__ */ jsx(
        TldrawUiButton,
        {
          type: "tool",
          "data-testid": `tools.${id}`,
          "aria-label": labelToUse,
          "data-value": id,
          onClick: () => onSelect("toolbar"),
          title: titleStr,
          onTouchStart: (e) => {
            preventDefault(e);
            onSelect("toolbar");
          },
          role: "radio",
          "aria-checked": isSelected ? "true" : "false",
          children: /* @__PURE__ */ jsx(TldrawUiButtonIcon, { icon })
        }
      );
    }
    case "toolbar-overflow": {
      return /* @__PURE__ */ jsx(TldrawUiDropdownMenuItem, { "aria-label": label, children: /* @__PURE__ */ jsx(
        TldrawUiButton,
        {
          type: "icon",
          className: "tlui-button-grid__button",
          onClick: () => {
            onSelect("toolbar");
          },
          "data-testid": `tools.more.${id}`,
          title: titleStr,
          role: "radio",
          "aria-checked": isSelected ? "true" : "false",
          "data-value": id,
          children: /* @__PURE__ */ jsx(TldrawUiButtonIcon, { icon })
        }
      ) });
    }
    default: {
      throw exhaustiveSwitchError(menuType);
    }
  }
}
export {
  TldrawUiMenuItem
};
//# sourceMappingURL=TldrawUiMenuItem.mjs.map
